import os
from flask import Flask, request, make_response, render_template, send_from_directory
from reportlab.pdfgen import canvas
from reportlab.lib.pagesizes import letter
from reportlab.lib.units import mm
from reportlab.lib import colors
import qrcode
from io import BytesIO
from datetime import datetime, timedelta

app = Flask(__name__)

# Asegúrate de tener una carpeta para guardar las placas generadas
OUTPUT_DIR = "static/placas_generadas"  # Cambié a una carpeta dentro de "static"
if not os.path.exists(OUTPUT_DIR):
    os.makedirs(OUTPUT_DIR)

# Ruta para servir los archivos generados (PDFs)
@app.route("/static/placas_generadas/<filename>")
def descargar_pdf(filename):
    return send_from_directory(OUTPUT_DIR, filename)

@app.route("/")
def index():
    return render_template("index.html")

@app.route("/generar_placa", methods=["POST"])
def generar_placa():
    try:
        # Datos del formulario
        placa = request.form.get("placa", "").upper()  # Recibimos la placa completa
        servicio = request.form.get("servicio", "")
        caducidad = int(request.form.get("caducidad", 30))  # Duración en días

        if "-" not in placa:
            return "El formato de la placa debe ser 'LETRAS-NÚMEROS'.", 400
        
        letras, numeros = placa.split("-")  # Separar letras y números de la placa

        if len(letras) != 3 or not letras.isalpha():
            return "Las letras deben ser exactamente 3 caracteres alfabéticos.", 400

        if len(numeros) not in [3, 4] or not numeros.isdigit():
            return "Los números deben ser de 3 o 4 dígitos.", 400

        # Generar el QR
        qr_data = f"Placa: {placa}, Organismo: GADM Puyango, Lugar y Fecha de Emisión: Alamor, {datetime.now().strftime("%d-%m-%Y")}"
        qr_img = qrcode.make(qr_data)

        # Guardar el QR en un archivo
        qr_file_path = os.path.join(OUTPUT_DIR, f"qr_{placa}.png")
        qr_img.save(qr_file_path)

        # Ruta absoluta de la imagen en la raíz
        fondo_path = os.path.join(os.getcwd(), "PP.png")  # Aquí se busca la imagen en la raíz
        if not os.path.exists(fondo_path):
            return f"Error: no se encuentra la imagen de fondo en la ruta {fondo_path}", 400

        # Generar el PDF
        placa_pdf_path = os.path.join(OUTPUT_DIR, f"placa_{placa}.pdf")
        c = canvas.Canvas(placa_pdf_path, pagesize=(300 * mm, 150 * mm))  # Tamaño 300mm x 150mm
        c.setFont("Helvetica", 12)

        # Fondo de la placa (utilizamos la imagen proporcionada)
        c.drawImage(fondo_path, 10 * mm, 10 * mm, width=280 * mm, height=130 * mm)  # Fondo de la placa

        # Centrar y separar el texto de la placa (letras-números)
        c.setFont("Helvetica-Bold", 165)  # Aumenté el tamaño de la fuente para hacer el texto más grande
        texto_placa = f"{letras}-{numeros}"
        # Calculamos el ancho del texto y lo centramos
        text_width = c.stringWidth(texto_placa, "Helvetica-Bold", 120)
        x_position = (230 * mm - text_width) / 2  # Centramos horizontalmente
        y_position = 53 * mm  # Ajustamos la posición para que esté más abajo
        c.drawString(x_position, y_position, texto_placa)  # Placa centrada y más abajo

        # Agregar el QR en la parte inferior (más abajo)
        c.drawImage(qr_file_path, 15 * mm, 5 * mm, width=30 * mm, height=30 * mm)

        # Agregar el texto del servicio centrado junto al QR, un poco más arriba
        c.setFont("Helvetica-Bold", 14)  # Aumenté el tamaño para que se vea más grande
        c.drawString(50 * mm, 20 * mm, f"Servicio: {servicio}")

        # Agregar la fecha de emisión y lugar (Alamor) en la parte inferior derecha
        fecha_emision = datetime.now().strftime("%d-%m-%Y")
        fecha_caducidad = datetime.now() + timedelta(days=caducidad)
        c.setFont("Helvetica", 10)
        c.drawString(180 * mm, 20 * mm, f"Lugar y fecha de emisión: Alamor, {fecha_emision}")
        c.drawString(180 * mm, 15 * mm, f"Fecha de caducidad: {fecha_caducidad.strftime('%d-%m-%Y')}")

        # Guardar el PDF generado
        c.showPage()
        c.save()

        # Devolver la ruta del PDF generado con el enlace
        return f"Placa generada con éxito. <a href='/static/placas_generadas/{os.path.basename(placa_pdf_path)}'>Descargar PDF</a>"

    except Exception as e:
        return f"Error al generar la placa: {e}", 500

if __name__ == "__main__":
    app.run(debug=True)
